<?php
require_once __DIR__ . '/config.php';

// Configurar manejo de errores
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

header('Content-Type: application/json');

try {
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    if ($method !== 'POST') {
        throw new Exception('Método no permitido');
    }

    // Validación básica
    function field($key) { return trim((string)($_POST[$key] ?? '')); }

    $name = field('name');
    $phone = field('phone');
    $email = field('email');
    $message = field('message');

    if ($name === '' || ($phone === '' && $email === '')) {
        throw new Exception('Nombre y al menos un medio de contacto son requeridos');
    }

    // Manejo de archivos simplificado
    $savedFiles = [];
    if (isset($_FILES['photos']) && !empty($_FILES['photos']['name'][0])) {
        $uploadsDir = __DIR__ . '/uploads';
        if (!is_dir($uploadsDir)) {
            @mkdir($uploadsDir, 0755, true);
        }
        
        $files = $_FILES['photos'];
        for ($i = 0; $i < count($files['name']); $i++) {
            if (empty($files['name'][$i])) continue;
            if ($files['error'][$i] !== UPLOAD_ERR_OK) continue;
            
            $tmpName = $files['tmp_name'][$i];
            $originalName = $files['name'][$i];
            
            // Validación básica
            if ($files['size'][$i] > 6 * 1024 * 1024) continue;
            
            $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
            if (!in_array($extension, ['jpg', 'jpeg', 'png', 'webp', 'pdf'])) continue;
            
            $newName = 'file_' . time() . '_' . $i . '.' . $extension;
            $destPath = $uploadsDir . '/' . $newName;
            
            if (@move_uploaded_file($tmpName, $destPath)) {
                $savedFiles[] = $newName;
            }
        }
    }

    // Envío de email con adjuntos
    $sent = false;
    if (function_exists('mail')) {
        $subject = 'Solicitud de Cotización de ' . $name;
        
        if (count($savedFiles) > 0) {
            // Email con adjuntos
            $boundary = '----=_NextPart_' . md5(time());
            
            $headers = [];
            $headers[] = 'From: ' . SMTP_USER;
            $headers[] = 'Reply-To: ' . ($email !== '' ? $email : SMTP_USER);
            $headers[] = 'MIME-Version: 1.0';
            $headers[] = 'Content-Type: multipart/mixed; boundary="' . $boundary . '"';
            
            $body = "--{$boundary}\r\n";
            $body .= "Content-Type: text/plain; charset=UTF-8\r\n";
            $body .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
            $body .= "Nombre: $name\r\n";
            $body .= "Teléfono: $phone\r\n";
            $body .= "Email: $email\r\n";
            $body .= "Mensaje: $message\r\n\r\n";
            $body .= "Archivos adjuntos: " . implode(', ', $savedFiles) . "\r\n";
            
            // Adjuntar archivos
            foreach ($savedFiles as $file) {
                $filePath = $uploadsDir . '/' . $file;
                if (file_exists($filePath)) {
                    $fileContent = file_get_contents($filePath);
                    $fileContent = chunk_split(base64_encode($fileContent));
                    
                    // Detectar tipo MIME básico
                    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                    $mimeType = 'application/octet-stream';
                    switch ($extension) {
                        case 'jpg':
                        case 'jpeg':
                            $mimeType = 'image/jpeg';
                            break;
                        case 'png':
                            $mimeType = 'image/png';
                            break;
                        case 'webp':
                            $mimeType = 'image/webp';
                            break;
                        case 'pdf':
                            $mimeType = 'application/pdf';
                            break;
                    }
                    
                    $body .= "\r\n--{$boundary}\r\n";
                    $body .= "Content-Type: {$mimeType}; name=\"{$file}\"\r\n";
                    $body .= "Content-Transfer-Encoding: base64\r\n";
                    $body .= "Content-Disposition: attachment; filename=\"{$file}\"\r\n\r\n";
                    $body .= $fileContent . "\r\n";
                }
            }
            
            $body .= "--{$boundary}--\r\n";
            
            $sent = @mail(EMAIL_TO, $subject, $body, implode("\r\n", $headers));
        } else {
            // Email simple sin archivos
            $bodyText = "Nombre: $name\nTeléfono: $phone\nEmail: $email\nMensaje: $message";
            
            $headers = [];
            $headers[] = 'From: ' . SMTP_USER;
            $headers[] = 'Reply-To: ' . ($email !== '' ? $email : SMTP_USER);
            $headers[] = 'MIME-Version: 1.0';
            $headers[] = 'Content-Type: text/plain; charset=UTF-8';
            
            $sent = @mail(EMAIL_TO, $subject, $bodyText, implode("\r\n", $headers));
        }
    }

    // Guardar respaldo
    $backupFile = __DIR__ . '/contact_backup.txt';
    $backupData = date('Y-m-d H:i:s') . "\n" . 
                  "Nombre: $name\n" . 
                  "Teléfono: $phone\n" . 
                  "Email: $email\n" . 
                  "Mensaje: $message\n" . 
                  "Archivos: " . implode(', ', $savedFiles) . "\n" . 
                  "Email enviado: " . ($sent ? 'Sí' : 'No') . "\n" . 
                  "---\n\n";
    @file_put_contents($backupFile, $backupData, FILE_APPEND | LOCK_EX);

    // WhatsApp link
    $waMsg = "Hola, soy $name. Tel: $phone. Correo: $email. Solicitud enviada desde la web Vasbal.";
    $waLink = 'https://wa.me/' . preg_replace('/\D+/', '', WHATSAPP_ALT_INTL) . '?text=' . urlencode($waMsg);

    // Respuesta exitosa
    echo json_encode([
        'ok' => true,
        'email_sent' => $sent,
        'whatsapp' => $waLink,
        'files' => $savedFiles,
        'message' => $sent ? 'Email enviado correctamente con ' . count($savedFiles) . ' archivo(s).' : 'Datos guardados. Usa WhatsApp para continuar.'
    ]);

} catch (Exception $e) {
    // Log del error
    error_log('Contact form error: ' . $e->getMessage());
    
    // Respuesta de error
    http_response_code(500);
    echo json_encode([
        'ok' => false,
        'error' => 'Error interno del servidor',
        'message' => 'Usa WhatsApp para continuar'
    ]);
}
?>



